﻿using Phidget22;
using Phidget22.ExampleUtils;
using System;
using System.Collections.Generic;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Windows.Forms;

namespace LEDArray_Example {
	public partial class Form1 : Form {
		Phidget22.LEDArray toOpen = null;
		Phidget22.LEDArray device = null;
		CommandLineOpen openArgs;
		private ErrorEventBox errorBox;
		formCleaner cleaner;
		bool ledState = false;
		LEDArrayColor[] fullStrand = null; //internal memory for what's being sent to the LED board
		int startLED, endLED;
		PictureBox[] ledDisplays; //container for all led visualizer pictureboxes
		Label[] displayLabels;
		List<PictureBox> visible = new List<PictureBox>(); //container for visible elements in ledDisplays
		List<Label> visibleLabels = new List<Label>(); //container for labels for visible elements in ledDisplays
		string patternFileName;
		bool animationMode = false;
		bool isHandCursor = false;
		LEDArrayAnimation[] animations = new LEDArrayAnimation[16];
		bool userDefinedAddressRange = false;
		bool userDefinedAddressRangeConfirm = false;

		private ColorPicker colorForm;

		public Form1() {
			InitializeComponent();
			openArgs = new CommandLineOpen(this);
			cleaner = new formCleaner(this);
			Size = PreferredSize;
		}

		public Form1(Phidget22.LEDArray ch) {
			InitializeComponent();
			toOpen = ch;
			cleaner = new formCleaner(this);
			Size = PreferredSize;
		}

		private void Form1_Load(object sender, EventArgs e) {
			errorBox = new ErrorEventBox(this);
			device = toOpen ?? openArgs.makeChannel<Phidget22.LEDArray>();

			device.Attach += device_attach;
			device.Detach += device_detach;
			device.Error += device_error;

			animationNumberCmb.SelectedIndex = 0;

			tabControl1.Size = new Size(381, 260);

			initializeValues();

			Enum[] supportedProtocols = new Enum[] {Phidget22.LEDArrayColorOrder.RGB,
													Phidget22.LEDArrayColorOrder.GRB,
													Phidget22.LEDArrayColorOrder.RGBW,
													Phidget22.LEDArrayColorOrder.GRBW};

			protoCombo.DataSource = supportedProtocols.ToArray<Enum>();
			protoCombo.SelectedIndexChanged += protoCombo_SelectedIndexChanged;

			Enum[] supportedAnimations = new Enum[] {Phidget22.LEDArrayAnimationType.ForwardScroll,
													 Phidget22.LEDArrayAnimationType.ReverseScroll,
													 Phidget22.LEDArrayAnimationType.ForwardScrollMirror,
													 Phidget22.LEDArrayAnimationType.ReverseScrollMirror,
													 Phidget22.LEDArrayAnimationType.Randomize };

			animTypeCombo.DataSource = supportedAnimations.ToArray<Enum>();

			try {
				device.Open();
			} catch (PhidgetException ex) { errorBox.addMessage("Error opening device: " + ex.Message); }

			colorForm = new ColorPicker(errorBox, this);
			UpdateDpiAwareness();
		}
		private void Form1_FormClosing(object sender, FormClosingEventArgs e) {
			device.Attach -= device_attach;
			device.Detach -= device_detach;
			device.Error -= device_error;

			device.Close();
		}
		public static System.Drawing.Image colorSpectrumImage {
			get {
				Assembly _assembly = Assembly.GetExecutingAssembly();
				String imageName = _assembly.GetManifestResourceNames().Where(img => img.EndsWith("colorSpectrum1.png")).FirstOrDefault();

				if (imageName == null)
					return null;

				return Image.FromStream(_assembly.GetManifestResourceStream(imageName));
			}
		}

		#region phidgetHandlers
		void device_attach(object sender, Phidget22.Events.AttachEventArgs e) {
			Phidget22.LEDArray attached = (Phidget22.LEDArray)sender;
			phidgetInfoBox1.FillPhidgetInfo(attached);

			configBox.Visible = true;
			tabControl1.Visible = true;

			attached.Gamma = 2.2; //magic number to make things look right on standard PC monitors

			endAddr.Value = 0;
			startAddr.Value = 0;

			basicEnd.Value = 0;
			basicStart.Value = 0;

			fullStrand = new LEDArrayColor[1];
			fullStrand[0].R = 0;
			fullStrand[0].G = 0;
			fullStrand[0].B = 0;


			ledState = true;

			try {
				device.Brightness = (double)(brightnessTrk.Value) / 100;
			} catch (PhidgetException ex) { errorBox.addMessage("Error setting brightness: " + ex.Message); }

			protoCombo.SelectedIndex = 1; //grb by default
			Size = PreferredSize;
		}

		void device_detach(object sender, Phidget22.Events.DetachEventArgs e) {
			configBox.Visible = false;
			tabControl1.Visible = false;
			Size = PreferredSize;
		}

		void device_error(object sender, Phidget22.Events.ErrorEventArgs e) {
			errorBox.addMessage(e.Description);
		}
		#endregion

		#region formControlHandlers
		private async void powerOn_Click(object sender, EventArgs e) {  //manual mode enable LEDs
			int fade = 0;
			try { fade = int.Parse(fadeTime.Text); } catch (Exception ex) { errorBox.addMessage("Error setting fade time: " + ex.Message); }
			if (fade > 10000) {
				fade = 10000;
				fadeTime.Text = 10000.ToString();
			}
			try {
				//pull led color info from datagridview
				fullStrand = null;
				fullStrand = new LEDArrayColor[patternBuilder.RowCount];
				for (int i = 0; i < patternBuilder.RowCount; i++) {
					DataGridViewCell tempCell = patternBuilder.Rows[i].Cells[1];
					if (tempCell is ColorPickerCell colorCell) {
						fullStrand[i].R = colorCell.ColorValue.R;
						fullStrand[i].G = colorCell.ColorValue.G;
						fullStrand[i].B = colorCell.ColorValue.B;
						fullStrand[i].W = (byte)colorCell.whiteValue;
					}
				}

				await device.SetLEDsAsync((int)startAddr.Value, (int)endAddr.Value, fullStrand, fade);
			} catch (PhidgetException ex) { errorBox.addMessage("Error setting LEDs: " + ex.Message); }
		}

		private void keySuppress(object sender, KeyEventArgs e) {
			if (e.KeyCode == Keys.Enter)
				e.SuppressKeyPress = true;
			userDefinedAddressRange = true;
		}

		private void enableBtn_Click(object sender, EventArgs e) {
			try {
				device.PowerEnabled = powerChk.Checked;
			} catch (PhidgetException ex) { errorBox.addMessage("Error enabling power: " + ex.Message); }
		}


		private void startAddr_ValueChanged(object sender, EventArgs e) {
			if (startAddr.Value < device.MinLEDCount - 1) {
				errorBox.addMessage("Error setting start address: Outside valid range " + device.MinLEDCount.ToString() + " - " + endAddr.Value.ToString());
				startAddr.Value = device.MinLEDCount;
			}
			if (startAddr.Value > endAddr.Value)
				endAddr.Value = startAddr.Value;
			if (userDefinedAddressRange)
				userDefinedAddressRangeConfirm = true;
		}

		private void endAddr_ValueChanged(object sender, EventArgs e) {
			if (endAddr.Value > device.MaxLEDCount || endAddr.Value < device.MinLEDCount - 1) { //make sure it's a valid count
				errorBox.addMessage("Error setting end address: Outside valid range " + device.MinLEDCount.ToString() + " - " + device.MaxLEDCount.ToString());
				endAddr.Value = device.MinLEDCount;
				return;
			}
			if (endAddr.Value < startAddr.Value)
				startAddr.Value = endAddr.Value;
			if (userDefinedAddressRange)
				userDefinedAddressRangeConfirm = true;
			if (fullStrand.Length != (int)endAddr.Value)
				fullStrand = new LEDArrayColor[(int)endAddr.Value];
		}

		private void ledRange_Click(object sender, EventArgs e) {
			userDefinedAddressRange = true;
		}

		private async void basicColor_Click(object sender, EventArgs e) {
			bool needsWhite = false;
			if (device.ColorOrder == LEDArrayColorOrder.RGBW || device.ColorOrder == LEDArrayColorOrder.GRBW)
				needsWhite = true;
			Color ColorValue;
			long whiteValue = 0;
			using (ColorPicker picker = new ColorPicker(errorBox, this, needsWhite)) {
				if (picker.ShowDialog() == DialogResult.OK) {
					ColorValue = picker.selection;
					whiteValue = picker.white;

					setColorBox(basicColor, basicRGB, ColorValue);

					if (!needsWhite)
						basicRGB.Text = $"#{ColorValue.R:X2}{ColorValue.G:X2}{ColorValue.B:X2}";
					else
						basicRGB.Text = $"#{ColorValue.R:X2}{ColorValue.G:X2}{ColorValue.B:X2}{whiteValue:X2}";

					LEDArrayColor[] single = new LEDArrayColor[1];
					single[0].R = (byte)(ColorValue.R);
					single[0].G = (byte)(ColorValue.G);
					single[0].B = (byte)(ColorValue.B);
					if (needsWhite)
						single[0].W = (byte)(whiteValue);

					int fade = 0;
					try { fade = int.Parse(fadeTime.Text); } catch (Exception ex) { errorBox.addMessage("Error setting fade time: " + ex.Message); }
					if (fade > 10000) {
						fade = 10000;
						fadeTime.Text = 10000.ToString();
					} else {
						try {
							await device.SetLEDsAsync((int)basicStart.Value, (int)basicEnd.Value, single, fade);
						} catch (PhidgetException ex) { errorBox.addMessage("Error setting brightness: " + ex.Message); }
					}
				}
			}
		}

		private void gammaTrk_Scroll(object sender, EventArgs e) {
			try {
				device.Gamma = (double)(gammaTrk.Value) / 10;
			} catch (PhidgetException ex) { errorBox.addMessage("Error setting gamme: " + ex.Message); }
			gammaTxt.Text = device.Gamma.ToString();
		}

		private void animStartBtn_Click(object sender, EventArgs e) {
			try {
				int colorColumn = 1;
				//pull led color info from datagridview
				fullStrand = null;
				fullStrand = new LEDArrayColor[patternBuilder.RowCount];
				for (int i = 0; i < patternBuilder.RowCount; i++) {
					DataGridViewCell tempCell = patternBuilder.Rows[i].Cells[colorColumn];
					if (tempCell is ColorPickerCell colorCell) {
						fullStrand[i].R = colorCell.ColorValue.R;
						fullStrand[i].G = colorCell.ColorValue.G;
						fullStrand[i].B = colorCell.ColorValue.B;
						fullStrand[i].W = (byte)colorCell.whiteValue;
					}
				}
			} catch (PhidgetException ex) { errorBox.addMessage("Error setting LEDs: " + ex.Message); }
			try {
				LEDArrayAnimation newAnimation = new LEDArrayAnimation();
				newAnimation.StartAddress = (short)int.Parse(startAnimTxt.Text);
				newAnimation.EndAddress = (short)(int.Parse(endAnimTxt.Text));
				newAnimation.Time = (short)int.Parse(timeAnimTxt.Text);
				newAnimation.AnimationType = (LEDArrayAnimationType)Enum.Parse(typeof(LEDArrayAnimationType), animTypeCombo.SelectedValue.ToString());
				try {
					device.SetAnimation(animationNumberCmb.SelectedIndex, fullStrand, newAnimation);
					animations[animationNumberCmb.SelectedIndex] = newAnimation;
				} catch (PhidgetException ex) { errorBox.addMessage("Error running animation: " + ex.Message); }
			} catch (Exception ex) { errorBox.addMessage("Error parsing animation info: " + ex.Message); }
		}

		private void stopAnim_Click(object sender, EventArgs e) {
			try {
				device.StopAnimation(animationNumberCmb.SelectedIndex);
			} catch (PhidgetException ex) { errorBox.addMessage("Error stopping animation: " + ex.Message); }
		}

		private void protoCombo_SelectedIndexChanged(object sender, EventArgs e) {
			if (device == null || ((System.Windows.Forms.ComboBox)sender).SelectedIndex == -1)
				return;

			try {
				device.ColorOrder = (LEDArrayColorOrder)Enum.Parse(typeof(LEDArrayColorOrder), protoCombo.SelectedValue.ToString());
			} catch (PhidgetException ex) { protoCombo.SelectedIndex = -1; errorBox.addMessage("Error setting protocol: " + ex.Message); }
		}

		private void brightnessTrk_ValueChanged(object sender, EventArgs e) {
			try {
				device.Brightness = (double)(brightnessTrk.Value) / 100;
			} catch (PhidgetException ex) { errorBox.addMessage("Error setting brightness: " + ex.Message); }
		}

		private void tabControl1_SelectedIndexChanged(object sender, EventArgs e) {
			if (tabControl1.SelectedIndex == 1) {
				tabControl1.Size = new Size(381, 605);
			} else {
				tabControl1.Size = new Size(381, 260);
			}
		}

		private void animationNumberCmb_SelectedIndexChanged(object sender, EventArgs e) {
			setAnimation(animationNumberCmb.SelectedIndex);

		}

		private void addRow_Click(object sender, EventArgs e) {
			patternBuilder.Rows.Add();
			patternBuilder.Rows[patternBuilder.Rows.Count - 1].Cells["Index"].Value = patternBuilder.Rows.Count - 1;
			if (!userDefinedAddressRangeConfirm && patternBuilder.Rows.Count > 1)
				endAddr.Value++;

		}

		private void removeRow_Click(object sender, EventArgs e) {
			if (patternBuilder.Rows.Count > 0)
				patternBuilder.Rows.RemoveAt(patternBuilder.Rows.Count - 1);
			if (!userDefinedAddressRangeConfirm && patternBuilder.Rows.Count > 0)
				endAddr.Value--;
		}

		#endregion


		#region helperFunctions
		private void setAnimation(int index) {
			LEDArrayAnimation selected = animations[index];

			startAnimTxt.Text = selected.StartAddress.ToString();
			endAnimTxt.Text = (selected.StartAddress + selected.EndAddress).ToString();
			timeAnimTxt.Text = selected.Time.ToString();

			switch (selected.AnimationType) {
			case LEDArrayAnimationType.ForwardScroll:
				animTypeCombo.SelectedIndex = 0;
				break;
			case LEDArrayAnimationType.ReverseScroll:
				animTypeCombo.SelectedIndex = 1;
				break;
			case LEDArrayAnimationType.ForwardScrollMirror:
				animTypeCombo.SelectedIndex = 2;
				break;
			case LEDArrayAnimationType.ReverseScrollMirror:
				animTypeCombo.SelectedIndex = 3;
				break;
			case LEDArrayAnimationType.Randomize:
				animTypeCombo.SelectedIndex = 4;
				break;
			default:
				break;
			}
		}

		private void checkVisibleLEDVisualizers() {
			int length = fullStrand.Length;
			int hundredsCount = 0;

			if (length > 0 && tabControl1.SelectedIndex == 1) {
				while (length > 100) {
					length -= 100;
					hundredsCount++;
				}

				for (int i = 0; i < hundredsCount; i++) {
					if (!ledDisplays[i].Visible) {
						ledDisplays[i].Visible = true;
						displayLabels[i * 2].Visible = true;
						displayLabels[i * 2 + 1].Visible = true;
						visible.Add(ledDisplays[i]);
						visibleLabels.Add(displayLabels[i * 2]);
						visibleLabels.Add(displayLabels[i * 2 + 1]);
					}
				}

				if (!ledDisplays[hundredsCount].Visible) {
					ledDisplays[hundredsCount].Visible = true;
					displayLabels[hundredsCount * 2].Visible = true;
					displayLabels[hundredsCount * 2 + 1].Visible = true;
					visible.Add(ledDisplays[hundredsCount]);
					visibleLabels.Add(displayLabels[hundredsCount * 2]);
					visibleLabels.Add(displayLabels[hundredsCount * 2 + 1]);
				}
				hundredsCount = 0;

			}
		}
		private void initializeValues() {

			if (endAddr.Value > 2048) {
				errorBox.addMessage("Error setting LED count: Max addressable LEDs is 2048");
				endAddr.Value = 2048;
				return;
			}

			fullStrand = new LEDArrayColor[(int)endAddr.Value];

			int length = fullStrand.Length;
			int hundredsCount = 0;

			if (length > 0 && tabControl1.SelectedIndex == 1) {

				while (length > 100) {
					length -= 100;
					hundredsCount++;
				}

				for (int i = 0; i < hundredsCount; i++) {
					ledDisplays[i].Visible = true;
					visible.Add(ledDisplays[i]);
				}

				ledDisplays[hundredsCount].Visible = true;
				visible.Add(ledDisplays[hundredsCount]);
				hundredsCount = 0;

			}
		}

		private void setColorBox(PictureBox toSet, Label label, Color color) {
			toSet.BackColor = color;
			label.Text = color.R.ToString("X2") + color.G.ToString("X2") + color.B.ToString("X2");
		}

		private void protoCombo_SelectedIndexChanged_1(object sender, EventArgs e) {
			if (protoCombo.SelectedIndex == -1)
				return;

			if ((LEDArrayColorOrder)Enum.Parse(typeof(LEDArrayColorOrder), protoCombo.SelectedValue.ToString()) == LEDArrayColorOrder.RGBW || (LEDArrayColorOrder)Enum.Parse(typeof(LEDArrayColorOrder), protoCombo.SelectedValue.ToString()) == LEDArrayColorOrder.GRBW) {
				if (patternBuilder.Columns[1] is ColorPickerColumn) {
					((ColorPickerColumn)patternBuilder.Columns[1]).setWRequired(true);
					patternBuilder.InvalidateColumn(1);
				}
			} else {
				if (patternBuilder.Columns[1] is ColorPickerColumn) {
					((ColorPickerColumn)patternBuilder.Columns[1]).setWRequired(false);
					patternBuilder.InvalidateColumn(1);
				}
			}
		}

		private void gammaTxt_KeyDown(object sender, KeyEventArgs e) {
			if (e.KeyCode == Keys.Enter) {
				setGamma(double.Parse(gammaTxt.Text));
				e.SuppressKeyPress = true;
			}
		}

		private void setGamma(double gamma) {
			try {
				device.Gamma = gamma;
			} catch (PhidgetException ex) { errorBox.addMessage("Error setting gamme: " + ex.Message); }
		}

		private void basicStart_ValueChanged(object sender, EventArgs e) {
			if (basicStart.Value > basicEnd.Value || basicStart.Value < device.MinLEDCount) {
				errorBox.addMessage("Error setting start address: Outside valid range " + device.MinLEDCount.ToString() + " - " + basicEnd.Value.ToString());
				basicStart.Value = device.MinLEDCount;
			}
		}

		private void basicEnd_ValueChanged(object sender, EventArgs e) {
			if (basicEnd.Value > device.MaxLEDCount || basicEnd.Value < device.MinLEDCount || basicEnd.Value < basicStart.Value) { //make sure it's a valid count
				errorBox.addMessage("Error setting end address: Outside valid range " + device.MinLEDCount.ToString() + " - " + device.MaxLEDCount.ToString());
				basicEnd.Value = basicStart.Value;
				return;
			}
		}

		private void brightnessTrk_Scroll(object sender, EventArgs e) {
			brightnessTxt.Text = brightnessTrk.Value.ToString() + "%";
		}

		private void Form1_DpiChanged(object sender, DpiChangedEventArgs e) {
			UpdateDpiAwareness();
		}

		private void UpdateDpiAwareness() {
			// Calculate DPI scale factor (assuming 96 DPI as baseline)
			float dpiScaleFactor = DeviceDpi / 96f;
			ColorPickerCell.DpiScaleFactor = dpiScaleFactor;

			// Adjust row heights based on DPI
			int scaledRowHeight = (int)(40 * dpiScaleFactor);
			patternBuilder.RowTemplate.Height = scaledRowHeight;

			if (patternBuilder.Rows.Count > 0) {
				patternBuilder.RowTemplate.Height = scaledRowHeight;
				for (int i = 0; i < patternBuilder.Rows.Count; i++) {
					patternBuilder.Rows[i].Height = scaledRowHeight;
				}
			}
		}

		private void patternBuilder_DpiChangedAfterParent(object sender, EventArgs e) {
			patternBuilder.PerformLayout();
			patternBuilder.Invalidate();
		}

		private void clear_Click(object sender, EventArgs e) {
			try {
				device.ClearLEDs();
			} catch (PhidgetException ex) { errorBox.addMessage("Error clearing: " + ex.Message); }
		}
		#endregion
	}


	#region extra classes
	public class HexNumericUpDown : System.Windows.Forms.NumericUpDown {
		public HexNumericUpDown() {
			Hexadecimal = true;
		}

		protected override void ValidateEditText() {
			if (base.UserEdit) {
				base.ValidateEditText();
			}
		}

		protected override void UpdateEditText() {
			Text = System.Convert.ToInt64(base.Value).ToString("X" + HexLength);
		}

		[System.ComponentModel.DefaultValue(4)]
		public int HexLength {
			get { return m_nHexLength; }
			set { m_nHexLength = value; }
		}

		public new System.Int64 Value {
			get { return System.Convert.ToInt64(base.Value); }
			set { base.Value = System.Convert.ToDecimal(value); }
		}

		private int m_nHexLength = 4;
	}

	public class ColorPickerCell : DataGridViewTextBoxCell {
		public Color ColorValue { get; set; } = Color.Black; //the color of the cell
		public long whiteValue = 0x00;
		public static float DpiScaleFactor { get; set; } = 1.0f;

		public override object Clone() {
			var clone = (ColorPickerCell)base.Clone();
			clone.ColorValue = this.ColorValue;
			return clone;
		}

		protected override void Paint(Graphics graphics, Rectangle clipBounds, Rectangle cellBounds, int rowIndex, DataGridViewElementStates elementState,
									  object value, object formattedValue, string errorText, DataGridViewCellStyle cellStyle,
									  DataGridViewAdvancedBorderStyle advancedBorderStyle, DataGridViewPaintParts paintParts) {

			base.Paint(graphics, clipBounds, cellBounds, rowIndex, elementState,
					   value, formattedValue, errorText, cellStyle, advancedBorderStyle,
					   paintParts & ~DataGridViewPaintParts.ContentForeground);

			// Calculate DPI-aware padding and spacing
			int padding = (int)(2 * DpiScaleFactor);
			int textOffset = (int)(15 * DpiScaleFactor);

			// Draw the color box
			Rectangle colorBox = new Rectangle(cellBounds.X + padding, cellBounds.Y + padding, cellBounds.Width - (2 * padding), cellBounds.Height - textOffset - padding);
			using (Brush brush = new SolidBrush(ColorValue)) {
				graphics.FillRectangle(brush, colorBox);
				graphics.DrawRectangle(Pens.Black, colorBox);
			}

			// Draw the hex value
			string hex;
			bool needsWhite = false;
			if (this.OwningColumn is ColorPickerColumn)
				needsWhite = ((ColorPickerColumn)this.OwningColumn).wRequired;
			if (!needsWhite)
				hex = $"#{ColorValue.R:X2}{ColorValue.G:X2}{ColorValue.B:X2}";
			else
				hex = $"#{ColorValue.R:X2}{ColorValue.G:X2}{ColorValue.B:X2}{whiteValue:X2}";
			TextRenderer.DrawText(graphics, hex, cellStyle.Font, new Point(cellBounds.X + padding, cellBounds.Bottom - textOffset), Color.Black);
		}

		protected override void OnClick(DataGridViewCellEventArgs e) {
			base.OnClick(e);
			bool needsWhite = false;
			if (this.OwningColumn is ColorPickerColumn)
				needsWhite = ((ColorPickerColumn)this.OwningColumn).wRequired;
			// Open your color picker form
			using (ColorPicker picker = new ColorPicker(needsWhite)) {
				picker.selection = ColorValue;

				if (picker.ShowDialog() == DialogResult.OK) {
					ColorValue = picker.selection;
					whiteValue = picker.white;
					if (!needsWhite)
						this.Value = $"#{ColorValue.R:X2}{ColorValue.G:X2}{ColorValue.B:X2}";
					else
						this.Value = $"#{ColorValue.R:X2}{ColorValue.G:X2}{ColorValue.B:X2}{whiteValue:X2}";
				}
				this.DataGridView.InvalidateCell(this);
			}
		}
	}

	public class ColorPickerColumn : DataGridViewColumn {
		public bool wRequired = false;

		public ColorPickerColumn() {
			this.CellTemplate = new ColorPickerCell();
		}

		public void setWRequired(bool value) {
			wRequired = value;
		}
	}
	#endregion
}
