﻿using Phidget22.ExampleUtils;
using System;
using System.Drawing;
using System.Windows.Forms;

namespace LEDArray_Example {
	public partial class ColorPicker : Form {

		ErrorEventBox errors;
		public Color selection;
		public long white;

		private Size baselineSelectorBoxSize;
		private Image baselineSelectorImage;
		private Font startingFont = null;
		private float startingDpi;

		public ColorPicker() {
			InitializeComponent();
			saveInitialDpiState();
		}

		public ColorPicker(ErrorEventBox errorBox, Form parent, bool showExtraControls) {
			InitializeComponent();
			saveInitialDpiState();
			wTrkPanel.Visible = showExtraControls;
			manualWPanel.Visible = showExtraControls;
			errors = errorBox;
		}

		public ColorPicker(bool showExtraControls) {
			InitializeComponent();
			saveInitialDpiState();
			wTrkPanel.Visible = showExtraControls;
			manualWPanel.Visible = showExtraControls;
		}

		public ColorPicker(ErrorEventBox errorBox, Form parent) {
			InitializeComponent();
			saveInitialDpiState();
			errors = errorBox;
		}

		private void ColorPicker_Load(object sender, EventArgs e) {
			manualR.Value = 0;
			manualG.Value = 0;
			manualB.Value = 0;
			manualW.Value = 0;

			ScaleDpiAware();
		}

		private void colorSelectorBox_MouseMove(object sender, MouseEventArgs e) {
			Bitmap pixelData = (Bitmap)colorSelectorBox.Image;
			try {
				Color clr = pixelData.GetPixel(e.X, e.Y);

				lblSmallScreen.BackColor = clr;

				lblRgbValues.Text = clr.R.ToString("X2") + clr.G.ToString("X2") + clr.B.ToString("X2");
			} catch (Exception ex) { }
		}

		private void colorSelectorBox_MouseUp(object sender, MouseEventArgs e) {
			Bitmap pixelData = (Bitmap)colorSelectorBox.Image;
			try {
				Color clr = pixelData.GetPixel(e.X, e.Y);

				selectedColor.BackColor = clr;
				manualR.Value = clr.R;
				manualG.Value = clr.G;
				manualB.Value = clr.B;
				selection = selectedColor.BackColor;
				white = manualW.Value;
			} catch (Exception ex) { }
		}

		private void colorSelectorTimeout_Tick(object sender, EventArgs e) {
			Rectangle colorSelectorLocation = new Rectangle(new Point(colorSelectorBox.Location.X, colorSelectorBox.Location.Y), colorSelectorBox.Size);
			if (!colorSelectorLocation.Contains(PointToClient(Control.MousePosition))) {
				lblSmallScreen.BackColor = SystemColors.Control;
				lblRgbValues.Text = String.Empty;
			}
		}

		private void ColorChange(object sender, EventArgs e) {
			selectedColor.BackColor = Color.FromArgb(0xff, (byte)manualR.Value, (byte)manualG.Value, (byte)manualB.Value);
			selection = selectedColor.BackColor;
			white = manualW.Value;
		}

		private void okBtn_Click(object sender, EventArgs e) {
			//check that a color has been selected
			selectedColor.BackColor = Color.FromArgb(0xff, (byte)manualR.Value, (byte)manualG.Value, (byte)manualB.Value);
			selection = selectedColor.BackColor;
			white = manualW.Value;
			if (selectedColor.BackColor == SystemColors.Control) {
				errors.addMessage("No color selected.");
				return;
			}
			this.DialogResult = DialogResult.OK;
			this.Close();
		}

		private void cancelBtn_Click(object sender, EventArgs e) {
			this.DialogResult = DialogResult.Cancel;
			this.Close();
		}

		private void ColorPicker_FormClosing(object sender, FormClosingEventArgs e) {
			ResetForm();
		}

		private void ResetForm() {
			lblSmallScreen.BackColor = SystemColors.Control;
			selectedColor.BackColor = SystemColors.Control;
			lblRgbValues.Text = String.Empty;
			wTrkPanel.Visible = false;
			manualWPanel.Visible = false;
		}

		private void ColorPicker_DpiChanged(object sender, DpiChangedEventArgs e) {
			ScaleDpiAware();
		}

		private void wTrk_MouseUp(object sender, MouseEventArgs e) {
			double multiplier = wTrk.Value / 100.0;
			manualW.Value = (long)(0xff * multiplier);
			white = manualW.Value;
		}

		private void manualW_ValueChanged(object sender, EventArgs e) {
			double trackValue = (double)manualW.Value / 0xff * 100;
			wTrk.Value = (int)trackValue;
			white = manualW.Value;
		}

		private void saveInitialDpiState() {

			// Store baseline sizes and image for DPI scaling
			baselineSelectorBoxSize = colorSelectorBox.Size;
			baselineSelectorImage = colorSelectorBox.Image;
			startingFont = lblRgbValues.Font;
			startingDpi = DeviceDpi;
		}

		private void ScaleDpiAware() {
			// Calculate DPI scale factor (assuming 96 DPI as baseline)
			float dpiScaleFactor = DeviceDpi / 96f;

			// Scale the color selector box
			if (baselineSelectorBoxSize != Size.Empty) {
				colorSelectorBox.Size = new Size(
					(int)(baselineSelectorBoxSize.Width * dpiScaleFactor),
					(int)(baselineSelectorBoxSize.Height * dpiScaleFactor)
				);
			}

			// Scale the image inside the picture box
			if (baselineSelectorImage != null) {
				int scaledWidth = (int)(baselineSelectorImage.Width * dpiScaleFactor);
				int scaledHeight = (int)(baselineSelectorImage.Height * dpiScaleFactor);
				Bitmap scaledImage = new Bitmap(baselineSelectorImage, new Size(scaledWidth, scaledHeight));
				
				// Dispose old scaled image if it exists and isn't the baseline
				if (colorSelectorBox.Image != null && colorSelectorBox.Image != baselineSelectorImage) {
					colorSelectorBox.Image.Dispose();
				}
				
				colorSelectorBox.Image = scaledImage;
			}

			float factor = DeviceDpi / startingDpi;
			lblRgbValues.Font = new Font(startingFont.FontFamily, startingFont.Size * factor, lblRgbValues.Font.Style);
		}
	}
}
